/*******************************************************************************
********************************************************************************
** COPYRIGHT:      (c) 2011 Rohde & Schwarz, Munich
** MODULE:         10GSock.cpp
** ABBREVIATION:   
** COMPILER:       Visual Studio 2005
** LANGUAGE:       C/C++
** AUTHOR:         Christian Poessinger
** ABSTRACT:       10G Socket abstraction class with winpcap library
** PREMISES:       
** REMARKS:        This program is distributed in the hope that it will be useful,
                   but WITHOUT ANY WARRANTY
** HISTORY:        
**	2011-03-31: (poessing)	Creation
** REVIEW:         
********************************************************************************/

/* INCLUDE FILES ***************************************************************/

/* IMPORT */
#include <iostream>
#include <cmath>
#include <complex>

/* EXPORT */
#include "10GSock.h"

/* GLOBAL VARIABLES DEFINITION *************************************************/

/* GLOBAL CONSTANTS DEFINITION *************************************************/

/* GLOBAL DEFINES **************************************************************/

/* LOCAL DEFINES ***************************************************************/
#ifdef _MSC_VER
#define _CRT_SECURE_NO_DEPRECATE
#endif

#define XG_FRAME_LEN    0x800
#define XG_FRAME_TYPE   0x02 // 16bit re + 16bit im
#define XG_DATA_HDR_LEN 0x13 

/* LOCAL TYPES DECLARATION *****************************************************/

/* LOCAL CLASSES DECLARATION ***************************************************/

/* LOCAL VARIABLES DEFINITION **************************************************/

/* LOCAL CONSTANTS DEFINITION **************************************************/
static const double pi = 3.141592653589793;

/* LOCAL FUNCTIONS DEFINITION **************************************************/

/* FUNCTION ********************************************************************/
C10GSock::C10GSock()

/*
SPECIFICATION:  ctor
PARAMETERS:
PRECONDITIONS:
SIDE_EFFECTS:
RETURN_VALUES:
EXCEPTIONS:
********************************************************************************/
{
    m_pHandle = NULL;
    m_pCurDevice = NULL;
    m_pAlldevs = NULL;

    m_nNumDevices = 0;
    m_IPAddress = 0;
    m_Netmask = 0;
    m_nMagicWordErr = 0;
    m_nFrameLenErr = 0;
    m_nFramesLost = 0;
    m_nFrameTypeErr = 0;
    m_nDataHdrLenErr = 0;
    m_nSigInval = 0;
    m_nBlanking = 0;
    m_lfRms = 0.0;
    m_nFrameCountOld = 0;
    m_nCurCount = 0;
    m_nTotalCount = 0;
    m_oldtime = 0;
}

/* FUNCTION ********************************************************************/
C10GSock::~C10GSock()

/*
SPECIFICATION:
PARAMETERS:
PRECONDITIONS:
SIDE_EFFECTS:
RETURN_VALUES:
EXCEPTIONS:
********************************************************************************/
{
    pcap_close(m_pHandle);
}

int C10GSock::FindAndSelectInterface(void)
{
  	int inum=0;
	int i=0;

    char errbuf[PCAP_ERRBUF_SIZE];

	/* Retrieve the device list */
	if(pcap_findalldevs(&m_pAlldevs, errbuf) == -1)
	{
		fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
		return -1;
	}
    
    /* Print the list */
    for(m_pCurDevice=m_pAlldevs; m_pCurDevice; m_pCurDevice=m_pCurDevice->next)
    {
        ++m_nNumDevices;
        printf("%d. %s", ++i, m_pCurDevice->name);
        if (m_pCurDevice->description)
            printf(" (%s)\n", m_pCurDevice->description);
        else
            printf(" (No description available)\n");
    }
	
    if(i==0)
    {
        printf("\nNo interfaces found! Make sure WinPcap is installed.\n");
        return -1;
    }
    
    printf("Enter the interface number (1-%d):",i);
    scanf("%d", &inum);
    
    if(inum < 1 || inum > i)
    {
        printf("\nInterface number out of range.\n");
        /* Free the device list */
        pcap_freealldevs(m_pAlldevs);
        return -1;
    }
	
    /* Jump to the selected adapter */
    for(m_pCurDevice=m_pAlldevs, i=0; i< inum-1 ;m_pCurDevice=m_pCurDevice->next, i++);

    return m_nNumDevices;
}

/* FUNCTION ********************************************************************/
 int C10GSock::CreateSocket( void )

/*
SPECIFICATION:
PARAMETERS:
PRECONDITIONS:
SIDE_EFFECTS:
RETURN_VALUES:
EXCEPTIONS:
********************************************************************************/
{
    int i=0;
    int res = 0;
    char errbuf[PCAP_ERRBUF_SIZE];
    char filter_exp[] = "ether proto 0x9A50";	/* only accept packets with proto 0x9A50 */
    struct bpf_program fp;		/* The compiled filter expression */
    typIFD_IFDATA_FRAME_EX *t_zfData;
  	struct pcap_pkthdr *header;
	const u_char *pkt_data;

    /* Open the device, Open the adapter */
    m_pHandle = pcap_open_live(m_pCurDevice->name,	// name of the device
                            9000,		// portion of the packet to capture. 
                            0,			// promiscuous mode (nonzero means promiscuous)
                            1000,		// read timeout
                            errbuf		// error buffer
                            );
    if (m_pHandle == NULL)
    {
        fprintf(stderr,"\nUnable to open the adapter. %s is not supported by WinPcap\n", m_pCurDevice->name);
        pcap_freealldevs(m_pAlldevs);
        return -1;
    }

    if (pcap_lookupnet(m_pCurDevice->name, &m_IPAddress, &m_Netmask, errbuf) == -1)
    {
        fprintf(stderr, "Can't get netmask for device %s\n", m_pCurDevice->name);
    }

    if (pcap_compile(m_pHandle, &fp, filter_exp, 0, m_IPAddress) == -1)
    {
        fprintf(stderr, "Couldn't parse filter %s: %s\n", filter_exp, pcap_geterr(m_pHandle));
        return(2);
    }

    if (pcap_setfilter(m_pHandle, &fp) == -1)
    {
         fprintf(stderr, "Couldn't install filter %s: %s\n", filter_exp, pcap_geterr(m_pHandle));
         return(2);
    }

    printf("\nlistening on %s...\n", m_pCurDevice->description);

    /* At this point, we don't need any more the device list. Free it */
    pcap_freealldevs(m_pAlldevs);
  
	/* Retrieve the packets */
	while((res = pcap_next_ex( m_pHandle, &header, &pkt_data)) >= 0)
    {
	    if(res == 0) /* Timeout elapsed */
			continue;
	
        t_zfData = (typIFD_IFDATA_FRAME_EX *)(pkt_data+SIZE_ETHERNET+AMMOS_ZF_DATA_OFFS);
        // test for magic word
        if (kFRH_MAGIC_WORD != t_zfData->frameheaderFrameHeader.uintMagicWord)
        {
            m_nMagicWordErr++;;
            continue;
        }
        // test for frame length
        if (XG_FRAME_LEN != t_zfData->frameheaderFrameHeader.uintFrameLength)
        {
            m_nFrameLenErr++;
            continue;
        }
        ParseIFData(t_zfData);
	}
	
	if(res == -1){
		printf("Error reading the packets: %s\n", pcap_geterr(m_pHandle));
		return -1;
	}
	
   return 0;
}

/* FUNCTION ********************************************************************/
 void C10GSock::ParseIFData(typIFD_IFDATA_FRAME_EX *data)

/*
SPECIFICATION: read the next packet from pcap
PARAMETERS:
    typIFD_IFDATA_FRAME_EX *data : pointer to IF data
PRECONDITIONS:
SIDE_EFFECTS:
RETURN_VALUES:
EXCEPTIONS:
********************************************************************************/
{
    unsigned long acttime;
    acttime = GetTickCount();

    if (XG_FRAME_TYPE != data->frameheaderFrameHeader.uintFrameType)
    {
        m_nFrameTypeErr++;
        return;
    }

    if (XG_DATA_HDR_LEN != data->frameheaderFrameHeader.uintDataHeaderLength)
    {
        m_nDataHdrLenErr++;
        return;
    }

    if (data->datablockDataBody->datablockheaderDatablockHeader.uintStatusword & kIFD_DATABLOCK_STATUSWORD__INVALID_BLOCK)
        m_nSigInval += data->ifdataheaderDataHeader.uintDatablockLength; // signal is invalid for ALL samples

    if (data->datablockDataBody->datablockheaderDatablockHeader.uintStatusword & kIFD_DATABLOCK_STATUSWORD__BLANKING)
        m_nBlanking += data->ifdataheaderDataHeader.uintDatablockLength; // blanking was active for ALL samples

    // Frame counts are incremented every frame. If there's a gap, we missed some frames
    if ((m_nFrameCountOld++ != data->frameheaderFrameHeader.uintFrameCount))
    {
        if (m_nTotalCount != 0) // only increase error count if it's not the first frame
            m_nFramesLost += (unsigned long)(data->frameheaderFrameHeader.uintFrameCount - m_nFrameCountOld);

        m_nFrameCountOld = data->frameheaderFrameHeader.uintFrameCount;
    }

    m_nCurCount++;
    m_nTotalCount++;

    // output data every second
    if ((acttime - m_oldtime) >= 1000)
    {
        unsigned long long ullFreq = data->ifdataheaderDataHeader.uintTunerFrequency_High;
        ullFreq <<= 32;
        ullFreq |= data->ifdataheaderDataHeader.uintTunerFrequency_Low;

        unsigned int nSampleRate = data->ifdataheaderDataHeader.uintSamplerate;

        unsigned long long ullStartTime = data->ifdataheaderDataHeader.bigtimeStartTimeStamp.structTimeInTwoWords.uintTime_HiOrderBits;
        ullStartTime <<= 32;
        ullStartTime |= data->ifdataheaderDataHeader.bigtimeStartTimeStamp.structTimeInTwoWords.uintTime_LoOrderBits;

        unsigned long long ullSampleCount = data->ifdataheaderDataHeader.uintSampleCounter_High;
        ullSampleCount <<= 32;
        ullSampleCount |= data->ifdataheaderDataHeader.uintSampleCounter_Low;

        long double llfSampleCount = (long double)ullSampleCount;
        llfSampleCount *= 1000000000;
        llfSampleCount /= nSampleRate;

        unsigned long long ullTimeStampCalc = ((unsigned long long)(llfSampleCount+0.5)) + ullStartTime;

        // pIQ points to IQ Samples 
        // data->ifdataheaderDataHeader.uintDatablockLength gives the number of complex samples
        // only for test purposes: calculate rms value over received IQ samples
        short *pIQ = (short *)data->datablockDataBody->uintData;
        double lfRms = 0.0;
        double lfFM = 0.0;
        double lfI;
        double lfQ;
        complex <double> oldSample, newSample;
        unsigned int i;
        for (i=0; i < data->ifdataheaderDataHeader.uintDatablockLength*2; i+=2)
        {
            // I Sample
            // INT16 -> double conversion
            lfI = pIQ[i];
            // norm to max. value 1.0
            lfI /= 32768.0                          /* 2^15 */;
            // Q Sample
            // INT16 -> double conversion
            lfQ = pIQ[i+1];
            // norm to max. value 1.0
            lfQ /= 32768.0                          /* 2^15 */;

            // now sum of power
            lfRms += lfI * lfI + lfQ * lfQ;

            newSample = complex<double>(lfI, lfQ);
            if (i > 0)
            {
                // used for frequency offset
                complex <double> cplxTemp;
                cplxTemp = newSample * conj(oldSample);
                lfFM += arg(cplxTemp)*nSampleRate/2/pi;
            }
            oldSample = newSample;
        }
        // RMS calculation 
        lfRms /= data->ifdataheaderDataHeader.uintDatablockLength;

        // Frequency offset calculation
        lfFM /= (data->ifdataheaderDataHeader.uintDatablockLength-1);

        // conversion to dBFS (full scale)
        lfRms = 10.0 * log10(lfRms);
        m_lfRms = lfRms;

        printf("===========================================\n");
        printf("Frames/sec                   =  %lu\n", m_nCurCount);
        printf("Overall frames recv.         =  %llu\n", m_nTotalCount); 
        printf("Frames lost                  =  %lu\n", m_nFramesLost);
        printf("Magic Word errors            =  %lu\n", m_nMagicWordErr);
        printf("Frame Type errors            =  %lu\n", m_nFrameTypeErr);
        printf("Header Length errors         =  %lu\n", m_nDataHdrLenErr);
        printf("RX frequency in Hz           =  %llu\n", ullFreq);
        printf("RX bandwidth in Hz           =  %lu\n", data->ifdataheaderDataHeader.uintBandwidth);
        printf("Samplerate in Samples/s      =  %lu\n", nSampleRate);
        printf("RMS level of last IF Frame   =  %.1lf dBFS, RMS ant. %.1lf dBuV\n", m_lfRms, m_lfRms+ 0.1*data->ifdataheaderDataHeader.intAntennaVoltageRef);
        printf("FM Offset in Hz              =  %.1lf\n", lfFM);
        printf("Starttime in ns              =  %llu\n", ullStartTime);
        printf("Timestamp in us              =  %llu\n", data->ifdataheaderDataHeader.bigtimeTimeStamp);
        printf("Timestamp Recalc in ns       =  %llu\n", ullTimeStampCalc);
        printf("Samplecount                  =  %llu\n", ullSampleCount);
        printf("KFactor                      =  0x%08x\n", data->ifdataheaderDataHeader.intKFactor);
        printf("Signal invalid for %lu samples\n", m_nSigInval);
        printf("Blanking active for %lu samples\n", m_nBlanking);

        m_nCurCount = 0;
        m_nSigInval = 0;
        m_nBlanking = 0;
        m_oldtime = acttime;
    }
}

/* GLOBAL FUNCTIONS DEFINITION *************************************************/

